<!doctype html>
<html lang="id">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>TLC Bites — Entry Stress</title>
<style>
  :root{
    --accent-green:#2ea153;
    --accent-yellow:#f5c23a;
    --bg:#ffffff;
    --muted-text:#6b7280;
    --danger:#d64545;
  }
  html,body{height:100%;margin:0;font-family:Inter,system-ui,Arial,sans-serif;background:linear-gradient(180deg,#fff 65%, #fbfff5 100%);}
  .decor{position:fixed;border-radius:50%;pointer-events:none;z-index:0;opacity:0.10}
  .decor.left{width:220px;height:220px;left:-80px;top:-80px;background:var(--accent-green)}
  .decor.right{width:180px;height:180px;right:-70px;bottom:-70px;background:var(--accent-yellow)}
  .wrap{max-width:420px;margin:0 auto;padding:18px;position:relative;z-index:1;min-height:100vh;display:flex;align-items:center;justify-content:center}
  .card{width:100%;background:var(--bg);border-radius:16px;box-shadow:0 10px 30px rgba(0,0,0,0.08);padding:16px;box-sizing:border-box}
  .progress{height:8px;background:#eee;border-radius:8px;overflow:hidden;margin-bottom:10px}
  .progressBar{height:100%;background:var(--accent-green);width:0%;transition:width .25s}
  .stepLabel{font-size:12px;color:var(--muted-text);text-align:center;margin-bottom:10px}
  .header{display:flex;gap:12px;align-items:center;margin-bottom:6px}
  .logo{width:44px;height:44px;border-radius:10px;background:linear-gradient(135deg,var(--accent-green),var(--accent-yellow));display:flex;align-items:center;justify-content:center;color:#fff;font-weight:800}
  .title{font-size:16px;font-weight:700;color:#222}
  .subtitle{font-size:13px;color:var(--muted-text)}
  .avatar{width:110px;height:110px;display:block;margin:10px auto;border-radius:12px;object-fit:contain;background:#fff}
  .qtitle{font-size:18px;color:#222;text-align:center;margin:10px 0}
  .qdesc{font-size:13px;color:var(--muted-text);text-align:center;margin-bottom:12px}
  .choices{display:flex;flex-direction:column;gap:10px;margin-top:6px}
  .opt{padding:12px;border-radius:10px;border:1px solid #eee;background:#fcfffb;color:#222;font-weight:700;text-align:center;cursor:pointer}
  .opt.selected{border-color:var(--accent-green);background:linear-gradient(180deg,#fff,#f0fff4);box-shadow:0 6px 20px rgba(46,161,83,0.08)}
  input[type=range]{width:100%;-webkit-appearance:none;height:6px;background:#eee;border-radius:6px}
  input[type=range]::-webkit-slider-thumb{-webkit-appearance:none;width:18px;height:18px;border-radius:50%;background:var(--accent-green);border:2px solid #333}
  .ticks{display:flex;justify-content:space-between;font-size:11px;color:var(--muted-text);margin-top:6px}
  .valueLarge{font-size:22px;font-weight:800;text-align:center;margin-top:8px}
  .controls{display:flex;gap:10px;margin-top:16px}
  .btn{flex:1;padding:12px;border-radius:10px;border:none;background:var(--accent-green);color:#fff;font-weight:700;cursor:pointer}
  .btn.light{background:#fff;color:var(--accent-green);border:1px solid #e6f3e9}
  .summaryWrap{max-height:260px;overflow-y:auto;margin-top:12px;padding-right:6px}
  .summaryItem{padding:12px;border-radius:10px;background:#fbfbfb;margin-bottom:8px;display:flex;justify-content:space-between;align-items:center}
  .resultBig{font-size:20px;font-weight:800;text-align:center;margin:8px 0}
  .badge{padding:6px 10px;border-radius:999px;color:#fff;font-weight:800}
  .badge.good{background:var(--accent-green)}
  .badge.mid{background:var(--accent-yellow);color:#000}
  .badge.bad{background:var(--danger)}
  /* calendar */
  .calendarWrap{margin-top:20px}
  .calendar{display:grid;grid-template-columns:repeat(7,1fr);gap:4px}
  .day{height:40px;display:flex;align-items:center;justify-content:center;
       border-radius:8px;background:#f9f9f9;font-size:18px}
  .day.today{border:2px solid var(--accent-green)}
  .day.empty{background:none}
</style>
</head>
<body>
  <div class="decor left"></div>
  <div class="decor right"></div>

  <div class="wrap">
    <div class="card">

      <!-- progress -->
      <div class="progress"><div id="progressBar" class="progressBar"></div></div>
      <div class="stepLabel" id="stepLabel">1 / 6</div>

      <!-- header -->
      <div class="header">
        <div class="logo">TLC</div>
        <div style="flex:1">
          <div class="title">Entry Data Stress</div>
          <div class="subtitle">Ceritakan sedikit tentang kondisi stresmu akhir-akhir ini</div>
        </div>
      </div>

      <!-- avatar -->
      <img id="avatar" class="avatar" src="cat-normal.gif" alt="avatar">

      <!-- content -->
      <div id="contentArea"></div>

      <!-- controls -->
      <div class="controls">
        <button id="backBtn" class="btn light" onclick="prevPage()" disabled>Back</button>
        <button id="nextBtn" class="btn" onclick="nextPage()">Next</button>
      </div>

    </div>
  </div>

<script>
const AVATARS = {
  normal: 'cat-normal.gif',
  mid: 'cat-warning.gif',
  stressed: 'cat-stressed.gif'
};

const pages = [
  {id:'welcome', type:'info', title:'Halo — Mari mulai', body:'Kami akan menanyakan beberapa hal singkat tentang tingkat stres dan kebiasaan menghadapinya.'},
  {id:'p_stress', type:'slider', title:'Seberapa besar tingkat stresmu akhir-akhir ini?', min:0, max:10, step:1, value:4, unit:'/10', desc:'0 = tidak stres, 10 = sangat stres'},
  {id:'symptoms', type:'options', title:'Seberapa sering kamu merasakan gejala stres?', options:['Jarang','Kadang','Sering','Hampir setiap hari']},
  {id:'sleep', type:'slider', title:'Bagaimana kualitas tidurmu?', min:0, max:10, step:1, value:6, unit:'/10', desc:'0 = sangat buruk, 10 = sangat baik'},
  {id:'coping', type:'multichoice', title:'Strategi coping yang sering kamu lakukan', options:['Olahraga','Ngobrol dengan teman','Makan berlebih','Alkohol/rokok','Meditasi','Mengabaikan masalah']},
  {id:'notes', type:'text', title:'Catatan tambahan (opsional)', placeholder:'Ceritakan hal lain...'}
];

let pageIndex=0;
const state={};
const contentArea=document.getElementById('contentArea');
const progressBar=document.getElementById('progressBar');
const stepLabel=document.getElementById('stepLabel');
const backBtn=document.getElementById('backBtn');
const nextBtn=document.getElementById('nextBtn');
const avatar=document.getElementById('avatar');

render();

function render(){
  contentArea.innerHTML='';
  const total=pages.length+1;
  progressBar.style.width=(pageIndex/(total-1))*100+'%';
  stepLabel.innerText=`${Math.min(pageIndex+1,total)} / ${total}`;
  if(pageIndex>=pages.length){renderSummary();return;}

  const p=pages[pageIndex];
  if(p.type==='info'){
    contentArea.innerHTML=`<div style="text-align:center"><h2 class="qtitle">${p.title}</h2><p class="qdesc">${p.body}</p></div>`;
    backBtn.disabled=true;nextBtn.disabled=false;avatar.src=AVATARS.normal;return;
  }
  if(p.type==='slider'){
    contentArea.innerHTML=`
      <div class="qtitle">${p.title}</div><div class="qdesc">${p.desc||''}</div>
      <input id="input-${p.id}" type="range" min="${p.min}" max="${p.max}" step="${p.step}" value="${p.value}">
      <div class="ticks"><span>${p.min}</span><span>${Math.round((p.min+p.max)/2)}</span><span>${p.max}</span></div>
      <div class="valueLarge" id="val-${p.id}">${p.value} ${p.unit||''}</div>
    `;
    const slider=document.getElementById(`input-${p.id}`);
    const valSpan=document.getElementById(`val-${p.id}`);
    slider.addEventListener('input',()=>{valSpan.textContent=slider.value+(p.unit||'');state[p.id]=Number(slider.value);updateAvatarByStress();});
    state[p.id]=Number(slider.value);updateAvatarByStress();
    backBtn.disabled=pageIndex===0;nextBtn.disabled=false;return;
  }
  if(p.type==='options'){
    const opts=document.createElement('div');opts.className='choices';
    p.options.forEach(opt=>{
      const b=document.createElement('div');b.className='opt';b.innerText=opt;
      if(state[p.id]===opt)b.classList.add('selected');
      b.onclick=()=>{Array.from(opts.children).forEach(c=>c.classList.remove('selected'));b.classList.add('selected');state[p.id]=opt;updateAvatarByStress();nextBtn.disabled=false;};
      opts.appendChild(b);
    });
    contentArea.innerHTML=`<div class="qtitle">${p.title}</div>`;contentArea.appendChild(opts);
    backBtn.disabled=false;nextBtn.disabled=state[p.id]===undefined;return;
  }
  if(p.type==='multichoice'){
    const opts=document.createElement('div');opts.className='choices';
    state[p.id]=state[p.id]||[];
    p.options.forEach(opt=>{
      const b=document.createElement('div');b.className='opt';b.innerText=opt;
      if(state[p.id].includes(opt))b.classList.add('selected');
      b.onclick=()=>{b.classList.toggle('selected');if(b.classList.contains('selected'))state[p.id].push(opt);else state[p.id]=state[p.id].filter(x=>x!==opt);updateAvatarByStress();};
      opts.appendChild(b);
    });
    contentArea.innerHTML=`<div class="qtitle">${p.title}</div>`;contentArea.appendChild(opts);
    backBtn.disabled=false;nextBtn.disabled=false;return;
  }
  if(p.type==='text'){
    contentArea.innerHTML=`<div class="qtitle">${p.title}</div><textarea id="txtnote" placeholder="${p.placeholder||''}" style="width:100%;min-height:100px;border-radius:10px;border:1px solid #eee;padding:10px"></textarea>`;
    if(state[p.id])document.getElementById('txtnote').value=state[p.id];
    backBtn.disabled=false;nextBtn.disabled=false;return;
  }
}

function nextPage(){if(pageIndex<pages.length){if(pages[pageIndex].type==='text')state['notes']=document.getElementById('txtnote').value;pageIndex++;render();}}
function prevPage(){if(pageIndex>0){pageIndex--;render();}}

function updateAvatarByStress(){
  const s=state['p_stress']||0;
  if(s<=3)avatar.src=AVATARS.normal;else if(s<=6)avatar.src=AVATARS.mid;else avatar.src=AVATARS.stressed;
}

function renderSummary(){
  // hitung skor sederhana
  const stress=state['p_stress']||0;
  let category='',badgeClass='',emo='';
  if(stress<=3){category='Low stress';badgeClass='good';emo='😊';avatar.src=AVATARS.normal;}
  else if(stress<=6){category='Moderate stress';badgeClass='mid';emo='😐';avatar.src=AVATARS.mid;}
  else {category='High stress';badgeClass='bad';emo='😣';avatar.src=AVATARS.stressed;}

  contentArea.innerHTML=`
    <div class="resultBig">Stress: <strong>${stress}/10</strong></div>
    <div style="text-align:center;margin-bottom:8px"><span class="badge ${badgeClass}">${category}</span></div>
    <div class="summaryWrap"><div id="summaryList"></div></div>
    <div class="calendarWrap"><h3 style="text-align:center;margin-top:16px;margin-bottom:8px">📅 My Mood Calendar</h3><div id="calendar" class="calendar"></div></div>
  `;
  const list=document.getElementById('summaryList');
  const add=(label,val)=>{const d=document.createElement('div');d.className='summaryItem';d.innerHTML=`<div><strong>${label}</strong><div style="font-size:13px;color:#444">${val||'-'}</div></div>`;list.appendChild(d);};
  add('Perceived stress',stress+'/10');add('Gejala',state['symptoms']);add('Tidur',state['sleep']);add('Coping',(state['coping']||[]).join(', '));add('Catatan',state['notes']||'-');
  backBtn.disabled=true;nextBtn.disabled=true;stepLabel.innerText='Selesai';progressBar.style.width='100%';
  saveTodayResult(category);renderCalendar();
}

/* calendar save & render */
function saveTodayResult(cat){
  const today=new Date();const key=today.toISOString().slice(0,10);
  localStorage.setItem('stress-'+key,cat);
}
function categoryEmoji(cat){
  if(cat.includes('Low'))return '😊';
  if(cat.includes('Moderate'))return '😐';
  if(cat.includes('High'))return '😣';
  return '❓';
}
function renderCalendar(){
  const cal=document.getElementById('calendar');cal.innerHTML='';
  const now=new Date();const year=now.getFullYear();const month=now.getMonth();
  const first=new Date(year,month,1);const last=new Date(year,month+1,0);
  const start=first.getDay();const days=last.getDate();
  for(let i=0;i<start;i++){cal.appendChild(Object.assign(document.createElement('div'),{className:'day empty'}));}
  for(let d=1;d<=days;d++){
    const date=new Date(year,month,d);const key=date.toISOString().slice(0,10);
    const cat=localStorage.getItem('stress-'+key);
    const cell=document.createElement('div');cell.className='day'+(d===now.getDate()?' today':'');
    cell.textContent=cat?categoryEmoji(cat):d;cal.appendChild(cell);
  }
}
</script>
</body>
</html>
